﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEngine.UI;

//GameAdvConfiguration is a script for managing the advanced configuration page UI
public class GameAdvConfiguration : MonoBehaviour
{
    //UI Elements
    public TMP_Dropdown CityTypesDropdown;
    public TMP_InputField SeedTextBox;
    public TMP_InputField PerlinOffsetTextBox;
    public Slider IterationsSlider;
    public Slider DensitySlider;

    private List<LSystem> _LSystems;

    public void Start()
    {
        //Get our known L-Systems from resources
        _LSystems = Resources.LoadAll("L-Systems", typeof(LSystem)).ToList().Cast<LSystem>().ToList();

        //Use the known L-Systems to populate the city types dropdown
        CityTypesDropdown.ClearOptions();
        foreach (LSystem system in _LSystems)
        {
            CityTypesDropdown.options.Add(new TMP_Dropdown.OptionData() { text = system.FriendlyName });
        }

        CityTypesDropdown.value = 1;
        CityTypesDropdown.value = 0;

        //Generate an initial seed and Perlin offset
        GenerateSeed();
        System.Random rand = new System.Random(Guid.NewGuid().GetHashCode());
        int off = rand.Next(0, ConfigurationManager.Instance.Generation.PerlinOffsetClamp);
        PerlinOffsetTextBox.text = off.ToString();
    }

    /// <summary>
    /// Plays the game by setting up the generation and beginning the loading process
    /// </summary>
    public void PlayGame()
    {
        GameController.Instance.LSystem = _LSystems[CityTypesDropdown.value];
        int offset = Convert.ToInt32(PerlinOffsetTextBox.text);
        GameController.Instance.PerlinOffset = new Vector2Int(offset, offset);
        GameController.Instance.LSystem.IterationsClamp = (int)IterationsSlider.value;
        GameController.Instance.CityDensity = (int)DensitySlider.value;
        GameController.Instance.CitySeed = Utilities.GetCitySeedIntFromString(SeedTextBox.text);

        AudioManager.Instance.StopMusic();
        Constants.LoadTask = new SceneLoadTask("GameScene", "LOADING GAME, PLEASE WAIT...");
        SceneManager.LoadScene("LoadingScene");
    }

    /// <summary>
    /// Generates a new city seed and displays it in the UI
    /// </summary>
    public void GenerateSeed()
    {
        SeedTextBox.text = Utilities.GenerateRandomCitySeed();
    }

    /// <summary>
    /// Sets the city types and its associated parameters based on the index into the list of city types
    /// </summary>
    /// <param name="index">The index of the city type to set</param>
    public void SetCityType(int index)
    {
        LSystem system = _LSystems[index];

        IterationsSlider.minValue = system.MinimumIterations;
        IterationsSlider.maxValue = system.MaximumIterations;
        IterationsSlider.value = IterationsSlider.minValue;
    }
}
